<?php
namespace WooBase\App;
use WooBase\Options\GlobalOptions;
use WooBase\Update\Updater;
use WooBase\Rest\RestLogout;
use WooBase\Rest\LemonSqueezy;

// Prevent direct access to this file
defined("ABSPATH") || exit();

/**
 * Class woobase
 *
 * Main class for initialising the woobase app.
 */
class woobase
{
  /**
   * woobase constructor.
   *
   * Initialises the main app.
   */
  public function __construct()
  {
    add_action("admin_menu", ["WooBase\App\WooBase", "admin_woobase_page"]);
    add_action("template_redirect", ["WooBase\App\WooBase", "check_for_frontend_page"]);

    // Starts new LemonSqueezy endpoint
    new GlobalOptions();
    new Updater();
    new RestLogout();
    new LemonSqueezy();
  }

  /**
   * Checks for frontend page
   *
   */
  public static function check_for_frontend_page()
  {
    $current_page_id = get_queried_object_id();

    $options = get_site_option("woobase_settings", []);
    $frontID =
      isset($options["load_on_page"]) && is_array($options["load_on_page"]) && isset($options["load_on_page"][0]) && $options["load_on_page"][0] != "" ? esc_html($options["load_on_page"][0]) : false;

    if (!$frontID) {
      return;
    }

    if ($current_page_id != $frontID) {
      return;
    }

    if (!is_user_logged_in()) {
      $current_url = home_url(add_query_arg([], $GLOBALS["wp"]->request));
      wp_redirect(wp_login_url($current_url));
      exit();
    }

    if (!current_user_can("manage_woocommerce") && !current_user_can("manage_options")) {
      wp_redirect(home_url());
      exit();
    }

    add_action("wp_enqueue_scripts", ["WooBase\App\WooBase", "load_styles"]);
    add_action("wp_footer", ["WooBase\App\WooBase", "build_woobase_frontend"], 1);
    add_action("wp_footer", ["WooBase\App\WooBase", "load_scripts"], 2);
  }

  // Checks for the shop manager role
  public static function is_shop_manager($user_id = null)
  {
    // If no user ID is provided, get the current user
    if (null === $user_id) {
      $user = wp_get_current_user();
    } else {
      $user = get_userdata($user_id);
    }

    // Check if we have a valid user object
    if (!$user || !$user->exists()) {
      return false;
    }

    error_log(json_encode($user->roles));
    // Check if the user has the 'shop_manager' role
    return in_array("shop_manager", (array) $user->roles);
  }

  /**
   * Adds woobase page.
   *
   * Calls add_menu_page to add new page .
   */
  public static function admin_woobase_page()
  {
    if (!current_user_can("manage_woocommerce") && !current_user_can("manage_options")) {
      return;
    }

    $icon =
      "data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIGhlaWdodD0iMjRweCIgdmlld0JveD0iMCAtOTYwIDk2MCA5NjAiIHdpZHRoPSIyNHB4IiBmaWxsPSIjZThlYWVkIj48cGF0aCBkPSJNMjQwLTgwcS0zMyAwLTU2LjUtMjMuNVQxNjAtMTYwdi00ODBxMC0zMyAyMy41LTU2LjVUMjQwLTcyMGg4MHEwLTY2IDQ3LTExM3QxMTMtNDdxNjYgMCAxMTMgNDd0NDcgMTEzaDgwcTMzIDAgNTYuNSAyMy41VDgwMC02NDB2NDgwcTAgMzMtMjMuNSA1Ni41VDcyMC04MEgyNDBabTAtODBoNDgwdi00ODBoLTgwdjgwcTAgMTctMTEuNSAyOC41VDYwMC01MjBxLTE3IDAtMjguNS0xMS41VDU2MC01NjB2LTgwSDQwMHY4MHEwIDE3LTExLjUgMjguNVQzNjAtNTIwcS0xNyAwLTI4LjUtMTEuNVQzMjAtNTYwdi04MGgtODB2NDgwWm0xNjAtNTYwaDE2MHEwLTMzLTIzLjUtNTYuNVQ0ODAtODAwcS0zMyAwLTU2LjUgMjMuNVQ0MDAtNzIwWk0yNDAtMTYwdi00ODAgNDgwWiIvPjwvc3ZnPg==";

    $options = get_site_option("woobase_settings", []);
    $menu_name = isset($options["menu_name"]) && $options["menu_name"] != "" ? esc_html($options["menu_name"]) : "WooBase";

    if (current_user_can("manage_woocommerce")) {
      $hook_suffix = add_menu_page($menu_name, $menu_name, "manage_woocommerce", "woobase", ["WooBase\App\WooBase", "build_woobase"], $icon);
    } elseif (current_user_can("manage_options")) {
      $hook_suffix = add_menu_page($menu_name, $menu_name, "manage_options", "woobase", ["WooBase\App\WooBase", "build_woobase"], $icon);
    }

    add_action("admin_head-{$hook_suffix}", ["WooBase\App\WooBase", "load_styles"]);
    add_action("admin_footer-{$hook_suffix}", ["WooBase\App\WooBase", "load_scripts"]);
  }

  /**
   * woobase page.
   *
   * Outputs the app holder for the frontend
   */
  public static function build_woobase_frontend()
  {
    // Enqueue the media library
    wp_enqueue_media();
    // Output the app
    echo "<div id='woobase-app' class='fixed inset-0 w-screen h-screen bg-white dark:bg-zinc-900' style='z-index:99999;position:fixed;top:0;left:0;width:100dvw;height:100dvh;'></div>";
  }

  /**
   * woobase page.
   *
   * Outputs the app holder
   */
  public static function build_woobase()
  {
    // Enqueue the media library
    wp_enqueue_media();
    // Output the app
    echo "<div id='woobase-app'></div>";
  }

  /**
   * woobase styles.
   *
   * Loads main lp styles
   */
  public static function load_styles()
  {
    // Get plugin url
    $url = plugins_url("woobase/");
    $style = $url . "app/dist/assets/styles/style.css";
    wp_enqueue_style("woobase", $style, [], woobase_plugin_version);
  }

  /**
   * woobase scripts.
   *
   * Loads main lp scripts
   */
  public static function load_scripts()
  {
    // Get plugin url
    $url = plugins_url("woobase/");
    $rest_base = get_rest_url();
    $rest_nonce = wp_create_nonce("wp_rest");
    $admin_url = get_admin_url();

    // User
    // Get the current user object
    $current_user = wp_get_current_user();
    $first_name = $current_user->first_name;

    // If first name is empty, fall back to display name
    if (empty($first_name)) {
      $first_name = $current_user->display_name;
    }

    // Get the user's email
    $email = $current_user->user_email;

    $frontPage = is_admin() ? "false" : "true";

    $manageOptions = current_user_can("manage_options") ? "true" : "false";

    $options = get_site_option("woobase_settings", false);
    $options = !$options ? [] : $options;
    // Remove the 'license_key' key
    unset($options["license_key"]);
    unset($options["instance_id"]);

    // Setup script object
    $builderScript = [
      "id" => "lmpress-script",
      "src" => $url . "app/dist/app.build.js?ver=" . woobase_plugin_version,
      "type" => "module",
      "plugin-base" => $url,
      "rest-base" => $rest_base,
      "rest-nonce" => $rest_nonce,
      "admin-url" => $admin_url,
      "user-id" => get_current_user_id(),
      "user-name" => $first_name,
      "user-email" => $email,
      "site-url" => get_home_url(),
      "front-page" => $frontPage,
      "can-manage-options" => $manageOptions,
      "woobase-settings" => json_encode($options),
    ];

    // Print tag
    wp_print_script_tag($builderScript);
  }
}
